/*******************************************************************************
********************************************************************************
** COPYRIGHT:    (c) 2004 Rohde & Schwarz, Munich
** MODULE:       wavfile.cpp
** ABBREVIATION:   
** COMPILER:     VC++ 5.0
** LANGUAGE:     C/C++
** AUTHOR:       Florian Behrens
** ABSTRACT:     
** PREMISES:     
** REMARKS:      
** HISTORY:      
**  2005-07-11: (Ob) Version 3.72: 8 Bit support added
**  2003-09-11: (FB) Version 3.71: Creation
** REVIEW:       
********************************************************************************/

/* INCLUDE FILES ***************************************************************/

/* IMPORT */
#include <memory.h>

/* EXPORT */
#include "wavfile.h"

/* LOCAL TYPES DECLARATION *****************************************************/

/* Strukturen fuer das WAV-Format */
struct struRIFFHeader
{
  char     cRIFFID[4];     // RIFF file id ('RIFF')
	unsigned unLength;       // Length in bytes after RIFF header
	char     cFormatID[4];   // Format id ('WAVE' for .WAV files)
};

struct struChunkHeader
{
  char     cChunkID[4];
  unsigned unChunkSize;    // Chunk length in bytes excluding header
};

struct struFormatChunkBody
{
  short          sFormatTag;
  unsigned short usChannels;
  unsigned       unSamplesPerSec;
  unsigned       unAvgBytesPerSec;
  short          sBlockAlign;
  short          sBitsPerSample;
  // Note: there may be additional fields here, depending upon wFormatTag
};

/* GLOBAL VARIABLES DEFINITION *************************************************/

/* GLOBAL CONSTANTS DEFINITION *************************************************/

/* GLOBAL DEFINES **************************************************************/

/* LOCAL DEFINES ***************************************************************/

/* flags for sFormatTag field of struFormatChunkBody */
#define WAVE_FORMAT_PCM 1

/* LOCAL TYPES DECLARATION *****************************************************/

/* LOCAL CLASSES DECLARATION ***************************************************/

/* LOCAL VARIABLES DEFINITION **************************************************/

/* LOCAL CONSTANTS DEFINITION **************************************************/

/* LOCAL FUNCTIONS DEFINITION **************************************************/

CWAVFile::CWAVFile()
: m_pcFilename(NULL),
  m_fFile(NULL),
  m_unFmtChunkPosition(0),
  m_unDataChunkPosition(0),
  m_unByteCount(0),
  m_unSampleRate(0),
  m_unChannelCount(0),
  m_unSampleWidth(0)
{}

CWAVFile::~CWAVFile()
{
  if (m_fFile)
    Close();
}

/***** FUNCTION *************************************************************/

bool CWAVFile::PrepareFile()

/****************************************************************************/
{
  /* Leeren Kopf der WAV-Datei als Platzhalter schreiben */
  struRIFFHeader ctRIFFHeader;
  if (fwrite(&ctRIFFHeader, sizeof(struRIFFHeader), 1, m_fFile) != 1)
    return false;

  /* Leeren 'fmt'-Chunk als Platzhalter schreiben */
  struChunkHeader ctChunckHeader;
  struFormatChunkBody ctFmtChunkBody;
  m_unFmtChunkPosition = ftell(m_fFile);
  if (fwrite(&ctChunckHeader, sizeof(struChunkHeader), 1, m_fFile) != 1)
    return false;
  if (fwrite(&ctFmtChunkBody, sizeof(struFormatChunkBody), 1, m_fFile) != 1)
    return false;
  
  /* Leeren 'data'-Chunk-Header als Platzhalter schreiben */
  m_unDataChunkPosition = ftell(m_fFile);
  memcpy(ctChunckHeader.cChunkID, "data", sizeof(ctChunckHeader.cChunkID));
  ctChunckHeader.unChunkSize = 0;
  if (fwrite(&ctChunckHeader, sizeof(struChunkHeader), 1, m_fFile) != 1)
    return false;

  return true;
}

/***** FUNCTION *************************************************************/

bool CWAVFile::PostprocWaveFile()

/****************************************************************************/
{
  fseek(m_fFile, 0, SEEK_SET);

  /* Kopf der WAV-Datei schreiben */
  struRIFFHeader ctRIFFHeader;
  memcpy(ctRIFFHeader.cRIFFID, "RIFF", sizeof(ctRIFFHeader.cRIFFID));
  ctRIFFHeader.unLength = sizeof(struRIFFHeader) - sizeof(ctRIFFHeader.cRIFFID) +
                          2 * sizeof(struChunkHeader) + sizeof(struFormatChunkBody) +
                          m_unByteCount;
  memcpy(ctRIFFHeader.cFormatID, "WAVE", sizeof(ctRIFFHeader.cFormatID));
  if (fwrite(&ctRIFFHeader, sizeof(struRIFFHeader), 1, m_fFile) != 1)
    return false;

  /* 'fmt'-Chunk schreiben */
  fseek(m_fFile, m_unFmtChunkPosition, SEEK_SET);
  struChunkHeader ctChunckHeader;
  struFormatChunkBody ctFmtChunkBody;
  memcpy(ctChunckHeader.cChunkID, "fmt ", sizeof(ctChunckHeader.cChunkID));
  ctChunckHeader.unChunkSize = sizeof(struFormatChunkBody);
  if (fwrite(&ctChunckHeader, sizeof(struChunkHeader), 1, m_fFile) != 1)
    return false;

  ctFmtChunkBody.sFormatTag       = WAVE_FORMAT_PCM;
  ctFmtChunkBody.usChannels       = m_unChannelCount; // 2-Kanal wg. I und Q
  ctFmtChunkBody.unSamplesPerSec  = m_unSampleRate;
  ctFmtChunkBody.sBlockAlign      = m_unSampleWidth; // 4 Bytes pro (komplexem) Abtastwert
  ctFmtChunkBody.unAvgBytesPerSec = m_unSampleRate * m_unSampleWidth;
  ctFmtChunkBody.sBitsPerSample   = 8*m_unSampleWidth/m_unChannelCount;
#if (0)
  if (ctFmtChunkBody.sBitsPerSample == 32)
  {
      // float !!!!! wegen MATLAB
      ctFmtChunkBody.sFormatTag = 3;

  }
#endif
  if (fwrite(&ctFmtChunkBody, sizeof(struFormatChunkBody), 1, m_fFile) != 1)
    return false;

  /* 'data'-Chunk-Header schreiben */
  fseek(m_fFile, m_unDataChunkPosition, SEEK_SET);
  memcpy(ctChunckHeader.cChunkID, "data", sizeof(ctChunckHeader.cChunkID));
  ctChunckHeader.unChunkSize = m_unByteCount;
  if (fwrite(&ctChunckHeader, sizeof(struChunkHeader), 1, m_fFile) != 1)
    return false;
  
  return true;
}

/***** FUNCTION *************************************************************/

bool CWAVFile::Open(unsigned unSampleRate, unsigned unChannelCount, 
                    unsigned unSampleWidth)

/****************************************************************************/
{
  if (m_fFile != NULL ||
      unSampleRate    == 0 || 
      (unChannelCount != 1 && unChannelCount != 2) || 
      (unSampleWidth  != 1 && unSampleWidth  != 2 && unSampleWidth  != 4 && unSampleWidth != 8))
    return false;

  /* Open WAV file for binary writing */
  m_fFile = fopen(m_pcFilename, "wb");
  if (m_fFile == NULL)
    return false;

  m_unByteCount = 0;

  m_unSampleRate   = unSampleRate;
  m_unChannelCount = unChannelCount;
  m_unSampleWidth  = unSampleWidth;

  return PrepareFile();
}

/***** FUNCTION *************************************************************/

bool CWAVFile::Close()

/****************************************************************************/
{
  bool bResult = PostprocWaveFile();
  fclose(m_fFile);
  m_fFile = NULL;

  return bResult;
}

/***** FUNCTION *************************************************************/

unsigned CWAVFile::write(void* pvData, unsigned unByteCount)

/****************************************************************************/
{
  unsigned unBytesWritten = 0;
  if (m_fFile)
  {
#if (0)
      if ((8*m_unSampleWidth/m_unChannelCount) == 32)
      {
          // umwandlung 32 Bit fixed format -> float format
          float *fTemp = new float[unByteCount/sizeof(float)];
          if (fTemp != NULL)
          {
              unsigned i;
              long *pLong = (long*)pvData;
              for (i=0; i<(unByteCount/sizeof(float)); i++)
              {
                  fTemp[i] = (float)pLong[i];
                  fTemp[i] /= 0x7FFFFFFF;
              }
              unBytesWritten = sizeof(float) * fwrite(fTemp, sizeof(float), i, m_fFile);
              delete fTemp;
          }
      }
      else
#endif
      {
        unBytesWritten = fwrite(pvData, sizeof(char), unByteCount, m_fFile);
      }
    m_unByteCount += unBytesWritten;
  }

  return unBytesWritten;
}

/***** FUNCTION *************************************************************/

void CWAVFile::SetFilename(char* pcFilename)

/****************************************************************************/
{
  m_pcFilename = pcFilename;
}


/***** FUNCTION *************************************************************/

char* CWAVFile::GetFilename() const

/****************************************************************************/
{
  return m_pcFilename;
}

/***** FUNCTION *************************************************************/

unsigned CWAVFile::GetSampleRate() const

/****************************************************************************/
{
  return m_unSampleRate;
}

/***** FUNCTION *************************************************************/

unsigned CWAVFile::GetChannelCount() const

/****************************************************************************/
{
  return m_unChannelCount;
}

/***** FUNCTION *************************************************************/

unsigned CWAVFile::GetSampleWidth() const

/****************************************************************************/
{
  return m_unSampleWidth;
}

/***** FUNCTION *************************************************************/

bool CWAVFile::IsOpen() const

/****************************************************************************/
{
  if (m_fFile)
    return true;
  return false;
}

/* End of file ***************************************************************/
